<?php
/**
 * Upload Manager pour Flying Book Creator v2.0
 * Gestion complète des uploads avec support multi-fichiers
 */

class FlyingBookUploadManager 
{
    private $uploadDir;
    private $allowedTypes;
    private $maxFileSize;
    private $pdo;
    
    public function __construct($projectId, $pdo) {
        $this->pdo = $pdo;
        $this->uploadDir = __DIR__ . '/uploads/projects/' . $projectId . '/';
        $this->allowedTypes = [
            'image' => ['jpg', 'jpeg', 'png', 'gif', 'webp'],
            'video' => ['mp4', 'webm', 'mov', 'avi'],
            'audio' => ['mp3', 'wav', 'ogg', 'm4a']
        ];
        $this->maxFileSize = 500 * 1024 * 1024; // 500MB
        
        // Créer les dossiers si nécessaire
        $this->ensureDirectories();
    }
    
    private function ensureDirectories() {
        $dirs = [
            $this->uploadDir,
            $this->uploadDir . 'images/',
            $this->uploadDir . 'videos/',
            $this->uploadDir . 'audio/',
            $this->uploadDir . 'thumbnails/'
        ];
        
        foreach ($dirs as $dir) {
            if (!file_exists($dir)) {
                mkdir($dir, 0755, true);
            }
        }
    }
    
    public function handleUpload($file, $projectId, $userId, $slideNumber = null) {
        try {
            // Validation de base
            if (!isset($file['tmp_name']) || !is_uploaded_file($file['tmp_name'])) {
                throw new Exception('Fichier non valide');
            }
            
            if ($file['size'] > $this->maxFileSize) {
                throw new Exception('Fichier trop volumineux (max 500MB)');
            }
            
            // Déterminer le type de fichier
            $fileInfo = $this->analyzeFile($file);
            
            // Générer un nom unique
            $newFileName = $this->generateUniqueFileName($fileInfo['extension']);
            
            // Chemin de destination
            $destPath = $this->uploadDir . $fileInfo['category'] . '/' . $newFileName;
            
            // Déplacer le fichier
            if (!move_uploaded_file($file['tmp_name'], $destPath)) {
                throw new Exception('Erreur lors du déplacement du fichier');
            }
            
            // Traitement post-upload
            $processedFile = $this->processFile($destPath, $fileInfo);
            
            // Enregistrer en base
            $mediaId = $this->saveToDatabase($projectId, $userId, $file, $newFileName, $fileInfo, $processedFile);
            
            return [
                'success' => true,
                'media_id' => $mediaId,
                'filename' => $newFileName,
                'url' => '/flying-book/uploads/projects/' . $projectId . '/' . $fileInfo['category'] . '/' . $newFileName,
                'thumbnail' => $processedFile['thumbnail'] ?? null,
                'duration' => $processedFile['duration'] ?? null,
                'dimensions' => $processedFile['dimensions'] ?? null
            ];
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    private function analyzeFile($file) {
        $extension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
        $mimeType = mime_content_type($file['tmp_name']);
        
        // Déterminer la catégorie
        $category = 'document';
        $fileType = 'document';
        
        foreach ($this->allowedTypes as $type => $extensions) {
            if (in_array($extension, $extensions)) {
                $category = $type === 'image' ? 'images' : ($type === 'video' ? 'videos' : 'audio');
                $fileType = $type;
                break;
            }
        }
        
        if ($fileType === 'document') {
            throw new Exception('Type de fichier non autorisé: ' . $extension);
        }
        
        return [
            'extension' => $extension,
            'mime_type' => $mimeType,
            'category' => $category,
            'file_type' => $fileType,
            'original_name' => $file['name'],
            'size' => $file['size']
        ];
    }
    
    private function generateUniqueFileName($extension) {
        return uniqid('fb_', true) . '_' . time() . '.' . $extension;
    }
    
    private function processFile($filePath, $fileInfo) {
        $result = [];
        
        switch ($fileInfo['file_type']) {
            case 'image':
                $result = $this->processImage($filePath, $fileInfo);
                break;
                
            case 'video':
                $result = $this->processVideo($filePath, $fileInfo);
                break;
                
            case 'audio':
                $result = $this->processAudio($filePath, $fileInfo);
                break;
        }
        
        return $result;
    }
    
    private function processImage($filePath, $fileInfo) {
        $result = [];
        
        // Obtenir les dimensions
        $imageInfo = getimagesize($filePath);
        if ($imageInfo) {
            $result['dimensions'] = $imageInfo[0] . 'x' . $imageInfo[1];
        }
        
        // Créer une miniature
        $result['thumbnail'] = $this->createImageThumbnail($filePath, $fileInfo);
        
        // Optimisation optionnelle
        $this->optimizeImage($filePath, $fileInfo);
        
        return $result;
    }
    
    private function processVideo($filePath, $fileInfo) {
        $result = [];
        
        // Créer une miniature vidéo
        $result['thumbnail'] = $this->createVideoThumbnail($filePath, $fileInfo);
        
        // Note: Pour obtenir la durée, il faudrait FFmpeg
        // $result['duration'] = $this->getVideoDuration($filePath);
        
        return $result;
    }
    
    private function processAudio($filePath, $fileInfo) {
        $result = [];
        
        // Note: Pour obtenir la durée, il faudrait getID3 ou FFmpeg
        // $result['duration'] = $this->getAudioDuration($filePath);
        
        return $result;
    }
    
    private function createImageThumbnail($filePath, $fileInfo) {
        try {
            $thumbnailDir = $this->uploadDir . 'thumbnails/';
            $thumbnailName = 'thumb_' . basename($filePath);
            $thumbnailPath = $thumbnailDir . $thumbnailName;
            
            // Dimensions de la miniature
            $thumbWidth = 300;
            $thumbHeight = 200;
            
            // Créer la miniature selon le type d'image
            $sourceImage = null;
            
            switch ($fileInfo['extension']) {
                case 'jpg':
                case 'jpeg':
                    $sourceImage = imagecreatefromjpeg($filePath);
                    break;
                case 'png':
                    $sourceImage = imagecreatefrompng($filePath);
                    break;
                case 'gif':
                    $sourceImage = imagecreatefromgif($filePath);
                    break;
                case 'webp':
                    $sourceImage = imagecreatefromwebp($filePath);
                    break;
            }
            
            if ($sourceImage) {
                $originalWidth = imagesx($sourceImage);
                $originalHeight = imagesy($sourceImage);
                
                // Calculer les nouvelles dimensions en gardant le ratio
                $ratio = min($thumbWidth / $originalWidth, $thumbHeight / $originalHeight);
                $newWidth = intval($originalWidth * $ratio);
                $newHeight = intval($originalHeight * $ratio);
                
                // Créer la nouvelle image
                $thumbnailImage = imagecreatetruecolor($newWidth, $newHeight);
                
                // Préserver la transparence pour PNG/GIF
                if ($fileInfo['extension'] === 'png' || $fileInfo['extension'] === 'gif') {
                    imagealphablending($thumbnailImage, false);
                    imagesavealpha($thumbnailImage, true);
                }
                
                // Redimensionner
                imagecopyresampled(
                    $thumbnailImage, $sourceImage,
                    0, 0, 0, 0,
                    $newWidth, $newHeight, $originalWidth, $originalHeight
                );
                
                // Sauvegarder
                imagejpeg($thumbnailImage, $thumbnailPath, 85);
                
                // Nettoyer la mémoire
                imagedestroy($sourceImage);
                imagedestroy($thumbnailImage);
                
                return 'thumbnails/' . $thumbnailName;
            }
            
        } catch (Exception $e) {
            // Log l'erreur mais ne pas faire planter l'upload
            error_log("Erreur création miniature: " . $e->getMessage());
        }
        
        return null;
    }
    
    private function createVideoThumbnail($filePath, $fileInfo) {
        // Pour créer des miniatures vidéo, il faudrait FFmpeg
        // Placeholder pour l'instant
        return null;
    }
    
    private function optimizeImage($filePath, $fileInfo) {
        // Optimisation optionnelle des images
        // Pourrait inclure compression, redimensionnement, etc.
    }
    
    private function saveToDatabase($projectId, $userId, $originalFile, $newFileName, $fileInfo, $processedFile) {
        $stmt = $this->pdo->prepare("
            INSERT INTO flying_book_media 
            (project_id, user_id, original_filename, stored_filename, file_path, 
             file_type, mime_type, file_size, dimensions, thumbnail_path, 
             upload_status, created_at) 
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'ready', NOW())
        ");
        
        $filePath = 'uploads/projects/' . $projectId . '/' . $fileInfo['category'] . '/' . $newFileName;
        
        $stmt->execute([
            $projectId,
            $userId,
            $originalFile['name'],
            $newFileName,
            $filePath,
            $fileInfo['file_type'],
            $fileInfo['mime_type'],
            $fileInfo['size'],
            $processedFile['dimensions'] ?? null,
            $processedFile['thumbnail'] ?? null
        ]);
        
        return $this->pdo->lastInsertId();
    }
    
    public function getProjectMedia($projectId) {
        $stmt = $this->pdo->prepare("
            SELECT * FROM flying_book_media 
            WHERE project_id = ? AND upload_status = 'ready'
            ORDER BY created_at DESC
        ");
        $stmt->execute([$projectId]);
        
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    public function deleteMedia($mediaId, $projectId, $userId) {
        try {
            // Récupérer les infos du fichier
            $stmt = $this->pdo->prepare("
                SELECT * FROM flying_book_media 
                WHERE id = ? AND project_id = ? AND user_id = ?
            ");
            $stmt->execute([$mediaId, $projectId, $userId]);
            $media = $stmt->fetch();
            
            if (!$media) {
                throw new Exception('Média non trouvé');
            }
            
            // Supprimer les fichiers physiques
            $mainFile = __DIR__ . '/' . $media['file_path'];
            if (file_exists($mainFile)) {
                unlink($mainFile);
            }
            
            if ($media['thumbnail_path']) {
                $thumbFile = $this->uploadDir . $media['thumbnail_path'];
                if (file_exists($thumbFile)) {
                    unlink($thumbFile);
                }
            }
            
            // Supprimer de la base
            $stmt = $this->pdo->prepare("DELETE FROM flying_book_media WHERE id = ?");
            $stmt->execute([$mediaId]);
            
            return ['success' => true];
            
        } catch (Exception $e) {
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }
}

// Utilisation dans flying_book_slides.php
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    $projectId = $_POST['project_id'] ?? null;
    $userId = $_SESSION['user_id'];
    
    switch ($_POST['action']) {
        case 'upload_file':
            if (!$projectId) {
                echo json_encode(['success' => false, 'error' => 'Project ID manquant']);
                exit;
            }
            
            if (!isset($_FILES['file'])) {
                echo json_encode(['success' => false, 'error' => 'Aucun fichier sélectionné']);
                exit;
            }
            
            $uploadManager = new FlyingBookUploadManager($projectId, $pdo);
            $result = $uploadManager->handleUpload($_FILES['file'], $projectId, $userId);
            
            echo json_encode($result);
            exit;
            
        case 'get_media':
            if (!$projectId) {
                echo json_encode(['success' => false, 'error' => 'Project ID manquant']);
                exit;
            }
            
            $uploadManager = new FlyingBookUploadManager($projectId, $pdo);
            $media = $uploadManager->getProjectMedia($projectId);
            
            echo json_encode(['success' => true, 'media' => $media]);
            exit;
            
        case 'delete_media':
            $mediaId = $_POST['media_id'] ?? null;
            if (!$mediaId || !$projectId) {
                echo json_encode(['success' => false, 'error' => 'IDs manquants']);
                exit;
            }
            
            $uploadManager = new FlyingBookUploadManager($projectId, $pdo);
            $result = $uploadManager->deleteMedia($mediaId, $projectId, $userId);
            
            echo json_encode($result);
            exit;
    }
}
?>